//+------------------------------------------------------------------+
//|                                                  Hunter_EA_v2.mq5 |
//|                                  Copyright 2024, MetaQuotes Ltd. |
//|                                             https://www.mql5.com |
//+------------------------------------------------------------------+
#property copyright "Copyright 2024, MetaQuotes Ltd."
#property link      "https://www.mql5.com"
#property version   "2.00"

#include <FVGLibrary.mqh>
#include <LiquidityLibrary.mqh>

//--- Input parameters
input group "=== Trading Settings ==="
input double InpLotSize = 0.01;                  // Lot size
input double InpProfitTarget = 2.0;              // Profit target in account currency ($)
input bool InpUseStopLoss = true;                // Use stop loss
input double InpStopLoss = 2.0;                  // Stop loss amount ($)

input group "=== FVG Detection Settings ==="
input ENUM_TIMEFRAMES InpFVGTimeframe = PERIOD_M5;  // FVG detection timeframe
input double InpMinGapSize = 50.0;               // Minimum gap size in points
input bool InpDrawFVGObjects = true;             // Draw FVG visual objects

input group "=== Trading Mode ==="
input bool InpUseEncroachmentMode = true;        // Use Encroachment Mode (waits for ENC cross)
input string InpModeNote = "OFF: Trade on any candle in FVG | ON: Trade only on ENC cross matching bias"; // Mode explanation

input group "=== Library Settings ==="
input int InpGMTOffset = 0;                      // GMT offset for sessions

//--- Global variables
CFVGLibrary* fvg_lib;
CLiquidityLibrary* liq_lib;
string daily_bias = "NEUTRAL";
ulong current_ticket = 0;
int last_traded_fvg_id = -1;                     // Track last FVG we traded

//--- EA Magic number
const int EA_MAGIC = 8889;

//+------------------------------------------------------------------+
//| Expert initialization function                                   |
//+------------------------------------------------------------------+
int OnInit()
{
   Print("=== Hunter EA v2 Initialized ===");
   
   // Initialize FVG Library with selected timeframe
   fvg_lib = new CFVGLibrary(_Symbol, InpFVGTimeframe, PERIOD_M1, 
                            InpMinGapSize, false, InpDrawFVGObjects);
   
   if(!fvg_lib.Init())
   {
      Print("Failed to initialize FVG Library!");
      return(INIT_FAILED);
   }
   fvg_lib.SetCallback(OnFVGEvent);
   
   // Initialize Liquidity Library
   liq_lib = new CLiquidityLibrary(_Symbol, PERIOD_M5, InpGMTOffset, false);
   
   if(!liq_lib.Init())
   {
      Print("Failed to initialize Liquidity Library!");
      return(INIT_FAILED);
   }
   
   // Get initial daily bias
   UpdateDailyBias();
   
   Print("=== CONFIGURATION ===");
   Print("FVG Timeframe: ", EnumToString(InpFVGTimeframe));
   Print("Min Gap Size: ", InpMinGapSize, " points");
   Print("Encroachment Mode: ", (InpUseEncroachmentMode ? "ON" : "OFF"));
   Print("Stop Loss: ", (InpUseStopLoss ? ("$" + DoubleToString(InpStopLoss, 2)) : "OFF"));
   Print("Profit Target: $", InpProfitTarget);
   Print("Daily Bias: ", daily_bias);
   Print("=====================");
   
   return(INIT_SUCCEEDED);
}

//+------------------------------------------------------------------+
//| Expert deinitialization function                                 |
//+------------------------------------------------------------------+
void OnDeinit(const int reason)
{
   if(fvg_lib != NULL)
   {
      fvg_lib.PrintFVGStats();
      fvg_lib.Deinit();
      delete fvg_lib;
      fvg_lib = NULL;
   }
   
   if(liq_lib != NULL)
   {
      liq_lib.PrintDailyBias();
      liq_lib.Deinit();
      delete liq_lib;
      liq_lib = NULL;
   }
   
   Print("=== Hunter EA v2 Deinitialized ===");
}

//+------------------------------------------------------------------+
//| Expert tick function                                             |
//+------------------------------------------------------------------+
void OnTick()
{
   // Update libraries
   if(fvg_lib != NULL) fvg_lib.OnTick();
   if(liq_lib != NULL) liq_lib.OnTick();
   
   // Update daily bias
   UpdateDailyBias();
   
   // Check for trading opportunities if no position open
   if(current_ticket == 0 || !PositionSelectByTicket(current_ticket))
   {
      if(!InpUseEncroachmentMode)
         CheckFVGEntry();
   }
   
   // Manage open position
   ManagePosition();
}

//+------------------------------------------------------------------+
//| Update daily bias from Liquidity Library                        |
//+------------------------------------------------------------------+
void UpdateDailyBias()
{
   if(liq_lib == NULL) return;
   
   static string last_bias = "";
   
   // Get daily bias from Liquidity Library
   daily_bias = liq_lib.GetDailyBias();
   
   // If neutral, try to get DOL direction
   if(daily_bias == "NEUTRAL")
   {
      string dol = liq_lib.GetCurrentDOL();
      
      if(dol == "PDH" || dol == "PWH" || StringFind(dol, "HIGH") != -1)
         daily_bias = "BULLISH";
      else if(dol == "PDL" || dol == "PWL" || StringFind(dol, "LOW") != -1)
         daily_bias = "BEARISH";
   }
   
   if(last_bias != daily_bias && last_bias != "")
   {
      Print("Daily bias updated: ", last_bias, " -> ", daily_bias);
   }
   last_bias = daily_bias;
}

//+------------------------------------------------------------------+
//| FVG Event Callback Function                                      |
//+------------------------------------------------------------------+
void OnFVGEvent(FVGInfo& fvg, string event_type)
{
   if(event_type == "FVG_FORMED")
   {
      Print("=== FVG FORMED ===");
      Print("ID: ", fvg.fvg_id, " | Type: ", (fvg.is_bullish ? "BULLISH" : "BEARISH"));
      Print("Gap: ", DoubleToString(fvg.top_price, _Digits), " - ", 
            DoubleToString(fvg.bottom_price, _Digits));
      Print("ENC Point: ", DoubleToString(fvg.enc_point, _Digits));
      Print("==================");
   }
   else if(event_type == "FVG_ENCROACHED" || event_type == "FVG_ENCROACHED_HA")
   {
      Print("=== FVG ENCROACHED ===");
      Print("ID: ", fvg.fvg_id, " | ENC Bias: ", fvg.directional_bias);
      Print("Daily Bias: ", daily_bias);
      Print("======================");
      
      // Handle encroachment mode trading
      if(InpUseEncroachmentMode)
      {
         HandleEncroachmentTrade(fvg);
      }
   }
   else if(event_type == "FVG_RESET")
   {
      Print("FVG ", fvg.fvg_id, " reset by inverse candle");
   }
}

//+------------------------------------------------------------------+
//| Check for FVG entry (non-encroachment mode)                      |
//+------------------------------------------------------------------+
void CheckFVGEntry()
{
   if(daily_bias == "NEUTRAL") return;
   
   double current_high = iHigh(_Symbol, InpFVGTimeframe, 0);
   double current_low = iLow(_Symbol, InpFVGTimeframe, 0);
   
   // Loop through active FVGs
   for(int i = 0; i < fvg_lib.GetFVGCount(); i++)
   {
      FVGInfo fvg = fvg_lib.GetFVGByIndex(i);
      
      if(!fvg.is_active || fvg.enc_touched) continue;
      if(fvg.fvg_id == last_traded_fvg_id) continue; // Skip already traded FVG
      
      // Check if current candle is inside FVG
      bool candle_in_fvg = false;
      
      if(fvg.is_bullish)
      {
         // Bullish FVG: check if candle touched the gap
         if(current_low <= fvg.top_price && current_low >= fvg.bottom_price)
            candle_in_fvg = true;
      }
      else
      {
         // Bearish FVG: check if candle touched the gap
         if(current_high >= fvg.bottom_price && current_high <= fvg.top_price)
            candle_in_fvg = true;
      }
      
      if(candle_in_fvg)
      {
         // Check if FVG type matches daily bias
         bool bias_matches = false;
         
         if(daily_bias == "BULLISH" && fvg.is_bullish)
            bias_matches = true;
         else if(daily_bias == "BEARISH" && !fvg.is_bullish)
            bias_matches = true;
         
         if(bias_matches)
         {
            Print("=== FVG ENTRY SIGNAL ===");
            Print("FVG ID: ", fvg.fvg_id);
            Print("FVG Type: ", (fvg.is_bullish ? "BULLISH" : "BEARISH"));
            Print("Daily Bias: ", daily_bias);
            Print("========================");
            
            OpenTradeForFVG(fvg);
            break; // Only one trade at a time
         }
      }
   }
}

//+------------------------------------------------------------------+
//| Handle encroachment mode trading                                 |
//+------------------------------------------------------------------+
void HandleEncroachmentTrade(FVGInfo& fvg)
{
   // Check if we already have a position
   if(current_ticket != 0 && PositionSelectByTicket(current_ticket))
   {
      Print("Position already open - cannot trade");
      return;
   }
   
   // Check if already traded this FVG
   if(fvg.fvg_id == last_traded_fvg_id)
   {
      Print("Already traded FVG ", fvg.fvg_id, " - skipping");
      return;
   }
   
   // Check if encroachment bias matches daily bias
   if(fvg.directional_bias != daily_bias)
   {
      Print("Encroachment bias (", fvg.directional_bias, 
            ") does not match daily bias (", daily_bias, ") - skipping");
      return;
   }
   
   Print("=== ENCROACHMENT MATCH ===");
   Print("FVG ID: ", fvg.fvg_id);
   Print("ENC Bias: ", fvg.directional_bias);
   Print("Daily Bias: ", daily_bias);
   Print("==========================");
   
   OpenTradeForFVG(fvg);
}

//+------------------------------------------------------------------+
//| Open trade for FVG                                                |
//+------------------------------------------------------------------+
void OpenTradeForFVG(FVGInfo& fvg)
{
   ulong ticket = 0;
   
   if(daily_bias == "BULLISH")
   {
      ticket = OpenBuy(InpLotSize);
   }
   else if(daily_bias == "BEARISH")
   {
      ticket = OpenSell(InpLotSize);
   }
   
   if(ticket > 0)
   {
      current_ticket = ticket;
      last_traded_fvg_id = fvg.fvg_id;
      fvg_lib.AssociateTradeWithFVG(fvg.fvg_id, ticket);
      
      Print("=== TRADE OPENED ===");
      Print("Ticket: ", ticket);
      Print("Direction: ", daily_bias);
      Print("FVG ID: ", fvg.fvg_id);
      Print("FVG Type: ", (fvg.is_bullish ? "BULLISH" : "BEARISH"));
      if(InpUseEncroachmentMode)
         Print("ENC Bias: ", fvg.directional_bias);
      Print("====================");
   }
}

//+------------------------------------------------------------------+
//| Manage open position                                             |
//+------------------------------------------------------------------+
void ManagePosition()
{
   if(current_ticket == 0) return;
   
   if(!PositionSelectByTicket(current_ticket))
   {
      // Position was closed externally
      Print("Position ", current_ticket, " closed externally");
      current_ticket = 0;
      return;
   }
   
   // Get current profit
   double profit = PositionGetDouble(POSITION_PROFIT);
   
   // Check profit target
   if(profit >= InpProfitTarget)
   {
      Print("=== PROFIT TARGET HIT ===");
      Print("Ticket: ", current_ticket);
      Print("Profit: $", DoubleToString(profit, 2));
      Print("=========================");
      
      if(PositionGetInteger(POSITION_TYPE) == POSITION_TYPE_BUY)
         CloseBuy(current_ticket);
      else
         CloseSell(current_ticket);
      
      current_ticket = 0;
      return;
   }
   
   // Check stop loss
   if(InpUseStopLoss && profit <= -InpStopLoss)
   {
      Print("=== STOP LOSS HIT ===");
      Print("Ticket: ", current_ticket);
      Print("Loss: $", DoubleToString(profit, 2));
      Print("=====================");
      
      if(PositionGetInteger(POSITION_TYPE) == POSITION_TYPE_BUY)
         CloseBuy(current_ticket);
      else
         CloseSell(current_ticket);
      
      current_ticket = 0;
   }
}

//+------------------------------------------------------------------+
//| Open Buy position                                                 |
//+------------------------------------------------------------------+
ulong OpenBuy(double lotSize)
{
   MqlTradeRequest request;
   MqlTradeResult result;
   ZeroMemory(request);
   ZeroMemory(result);
   
   request.action = TRADE_ACTION_DEAL;
   request.symbol = _Symbol;
   request.volume = lotSize;
   request.type = ORDER_TYPE_BUY;
   request.price = SymbolInfoDouble(_Symbol, SYMBOL_ASK);
   request.deviation = 20;
   request.type_filling = ORDER_FILLING_FOK;
   request.magic = EA_MAGIC;
   
   if(OrderSend(request, result))
   {
      if(result.deal > 0)
      {
         Print("Buy order opened successfully");
         return result.deal;
      }
   }
   else
   {
      Print("Failed to open buy order. Error: ", GetLastError());
   }
   
   return 0;
}

//+------------------------------------------------------------------+
//| Open Sell position                                                |
//+------------------------------------------------------------------+
ulong OpenSell(double lotSize)
{
   MqlTradeRequest request;
   MqlTradeResult result;
   ZeroMemory(request);
   ZeroMemory(result);
   
   request.action = TRADE_ACTION_DEAL;
   request.symbol = _Symbol;
   request.volume = lotSize;
   request.type = ORDER_TYPE_SELL;
   request.price = SymbolInfoDouble(_Symbol, SYMBOL_BID);
   request.deviation = 20;
   request.type_filling = ORDER_FILLING_FOK;
   request.magic = EA_MAGIC;
   
   if(OrderSend(request, result))
   {
      if(result.deal > 0)
      {
         Print("Sell order opened successfully");
         return result.deal;
      }
   }
   else
   {
      Print("Failed to open sell order. Error: ", GetLastError());
   }
   
   return 0;
}

//+------------------------------------------------------------------+
//| Close Buy position                                                |
//+------------------------------------------------------------------+
void CloseBuy(ulong ticket)
{
   if(!PositionSelectByTicket(ticket)) return;
   
   double volume = PositionGetDouble(POSITION_VOLUME);
   double price = SymbolInfoDouble(_Symbol, SYMBOL_BID);
   
   MqlTradeRequest request;
   MqlTradeResult result;
   ZeroMemory(request);
   ZeroMemory(result);
   
   request.action = TRADE_ACTION_DEAL;
   request.symbol = _Symbol;
   request.volume = volume;
   request.type = ORDER_TYPE_SELL;
   request.price = price;
   request.deviation = 20;
   request.type_filling = ORDER_FILLING_FOK;
   request.position = ticket;
   request.magic = EA_MAGIC;
   
   if(OrderSend(request, result))
   {
      Print("Buy position closed successfully");
   }
   else
   {
      Print("Failed to close buy position. Error: ", GetLastError());
   }
}

//+------------------------------------------------------------------+
//| Close Sell position                                               |
//+------------------------------------------------------------------+
void CloseSell(ulong ticket)
{
   if(!PositionSelectByTicket(ticket)) return;
   
   double volume = PositionGetDouble(POSITION_VOLUME);
   double price = SymbolInfoDouble(_Symbol, SYMBOL_ASK);
   
   MqlTradeRequest request;
   MqlTradeResult result;
   ZeroMemory(request);
   ZeroMemory(result);
   
   request.action = TRADE_ACTION_DEAL;
   request.symbol = _Symbol;
   request.volume = volume;
   request.type = ORDER_TYPE_BUY;
   request.price = price;
   request.deviation = 20;
   request.type_filling = ORDER_FILLING_FOK;
   request.position = ticket;
   request.magic = EA_MAGIC;
   
   if(OrderSend(request, result))
   {
      Print("Sell position closed successfully");
   }
   else
   {
      Print("Failed to close sell position. Error: ", GetLastError());
   }
}
//+------------------------------------------------------------------+